﻿# Fonction utilitaire pour vérifier la présence d'un programme installé via le registre
function Test-ProgramInstalled {
    param(
        [string]$DisplayName
    )
    $paths = @(
        'HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall',
        'HKLM:\SOFTWARE\WOW6432Node\Microsoft\Windows\CurrentVersion\Uninstall'
    )
    foreach ($regPath in $paths) {
        $apps = Get-ChildItem $regPath | ForEach-Object {
            Get-ItemProperty $_.PSPath
        }
        foreach ($app in $apps) {
            if ($app.DisplayName -and $app.DisplayName -like "*$DisplayName*") {
                return $true
            }
        }
    }
    return $false
}


# Dossier courant du script
$InstallPath = $PSScriptRoot
if (-not (Test-Path "$InstallPath\Logging")) {
    New-Item -Path $InstallPath -Name "Logging" -ItemType Directory | Out-Null
}

# URLs des prérequis
$VC2012Uri = "https://download.microsoft.com/download/1/6/b/16b06f60-3b20-4ff2-b699-5e9b7962f9ae/VSU_4/vcredist_x64.exe"
$VC2013Uri = "https://download.visualstudio.microsoft.com/download/pr/10912041/cee5d6bca2ddbcd039da727bf4acb48a/vcredist_x64.exe"
#$UCMAUri = "https://download.microsoft.com/download/2/c/4/2c47a5c1-a1f3-4843-b9fe-84c0032c61ec/UcmaRuntimeSetup.exe"
$URLRWUri = "https://download.microsoft.com/download/1/2/8/128E2E22-C1B9-44A4-BE2A-5859ED1D4592/rewrite_amd64_en-US.msi"

# Chemins des exécutables
$Ucma      = Join-Path $InstallPath "UCMARedist\Setup.exe"
$VcRedist2012 = Join-Path $InstallPath "VC2012x64.exe"
$VcRedist2013 = Join-Path $InstallPath "VC2013x64.exe"
$Rewrite   = Join-Path $InstallPath "rewrite_amd64_en-US.msi"

# Téléchargement si absent
if (-not (Test-Path $VcRedist2012)) {
    Write-Host "Téléchargement VC++ 2012..."
    Invoke-WebRequest -Uri $VC2012Uri -OutFile $VcRedist2012
}
if (-not (Test-Path $VcRedist2013)) {
    Write-Host "Téléchargement VC++ 2013..."
    Invoke-WebRequest -Uri $VC2013Uri -OutFile $VcRedist2013
}
if (-not (Test-Path $Ucma)) {
    Write-Host "Téléchargement UCMA Runtime..."
    Invoke-WebRequest -Uri $UCMAUri -OutFile $Ucma
}
if (-not (Test-Path $Rewrite)) {
    Write-Host "Téléchargement IIS URL Rewrite..."
    Invoke-WebRequest -Uri $URLRWUri -OutFile $Rewrite
}

# Vérification de l’existence des fichiers
$Files = @($Ucma, $VcRedist2012, $VcRedist2013, $Rewrite)
foreach ($File in $Files) {
    if (-not (Test-Path $File)) {
        Write-Error "Fichier manquant : $File ($InstallPath)"
        exit 1
    }
}

Write-Host "Tous les fichiers nécessaires sont présents." -ForegroundColor Green


# Installation des rôles et fonctionnalités Windows Server Core pour Exchange Mailbox
Write-Host "Installation des rôles et fonctionnalités Exchange Server SE..." -ForegroundColor Yellow
$osVersion = (Get-CimInstance Win32_OperatingSystem).Version
$featureList = @(
    'Server-Media-Foundation', 'NET-Framework-45-Core', 'NET-Framework-45-ASPNET', 'NET-WCF-HTTP-Activation45',
    'NET-WCF-Pipe-Activation45', 'NET-WCF-TCP-Activation45', 'NET-WCF-TCP-PortSharing45', 'RPC-over-HTTP-proxy',
    'RSAT-Clustering', 'RSAT-Clustering-CmdInterface', 'RSAT-Clustering-PowerShell', 'WAS-Process-Model',
    'Web-Asp-Net45', 'Web-Basic-Auth', 'Web-Client-Auth', 'Web-Digest-Auth', 'Web-Dir-Browsing', 'Web-Dyn-Compression',
    'Web-Http-Errors', 'Web-Http-Logging', 'Web-Http-Redirect', 'Web-Http-Tracing', 'Web-ISAPI-Ext', 'Web-ISAPI-Filter',
    'Web-Metabase', 'Web-Mgmt-Console', 'Web-Mgmt-Service', 'Web-Net-Ext45', 'Web-Request-Monitor', 'Web-Server',
    'Web-Stat-Compression', 'Web-Static-Content', 'Web-Filtering', 'Web-Windows-Auth', 'Web-WMI', 'RSAT-ADDS', 'GPMC', 'RSAT-DNS-Server'
)
if ([version]$osVersion -ge [version]'10.0.25398') {
    Write-Host "Windows Server 2025 détecté : Windows-Identity-Foundation sera retiré de la liste des rôles." -ForegroundColor Yellow
} else {
    $featureList += 'Windows-Identity-Foundation'
}
Install-WindowsFeature $featureList -ErrorAction Stop | Out-Null

# Suppression optionnelle de MSMQ (non requis pour Exchange)
Write-Host "Suppression optionnelle de MSMQ..." -ForegroundColor Yellow
Remove-WindowsFeature NET-WCF-MSMQ-Activation45, MSMQ


# ==========================
# ETAPE : Installation des packages requis (Visual C++, UCMA, IIS Rewrite)
# ==========================

# Fonction utilitaire pour vérifier la présence des DLL après installation
function Test-DllPresence {
    param(
        [string[]]$DllNames
    )
    $allPresent = $true
    foreach ($dll in $DllNames) {
        $dllPath = Join-Path $env:windir "System32\$dll"
        if (-not (Test-Path $dllPath)) {
            Write-Host "[WARNING] DLL manquante : $dllPath" -ForegroundColor Yellow
            $allPresent = $false
        }
    }
    return $allPresent
}

# --- VC++ 2012 ---
if (Test-ProgramInstalled 'Visual C++ 2012') {
    Write-Host "[SKIP] VC++ 2012 déjà installé. Installation ignorée."
} else {
    Write-Host "[Core] Installation de Visual C++ 2012 Redistributable..." -ForegroundColor Cyan
    $objProc = Start-Process -FilePath $VcRedist2012 -ArgumentList "/passive /norestart" -NoNewWindow -PassThru -Wait
    Write-Host "Code retour VC++ 2012 : $($objProc.ExitCode)"
    if ($objProc.ExitCode -notmatch "^(0|1638|3010)$") {
        Write-Host "[ERREUR] VC++ 2012 : L'installation a échoué (code $($objProc.ExitCode))." -ForegroundColor Red
        pause
    } else {
        if (Test-DllPresence @('msvcr110.dll','msvcp110.dll')) {
            Write-Host "[OK] VC++ 2012 : DLL présentes."
        } else {
            Write-Host "[WARNING] VC++ 2012 : DLL manquantes."
        }
        if (Test-ProgramInstalled 'Visual C++ 2012') {
            Write-Host "[OK] VC++ 2012 détecté dans les programmes installés."
        } else {
            Write-Host "[WARNING] VC++ 2012 NON détecté dans les programmes installés." -ForegroundColor Yellow
        }
    }
}

# --- VC++ 2013 ---
if (Test-ProgramInstalled 'Visual C++ 2013') {
    Write-Host "[SKIP] VC++ 2013 déjà installé. Installation ignorée."
} else {
    Write-Host "[Core] Installation de Visual C++ 2013 Redistributable..." -ForegroundColor Cyan
    $objProc = Start-Process -FilePath $VcRedist2013 -ArgumentList "/passive /norestart" -NoNewWindow -PassThru -Wait
    Write-Host "Code retour VC++ 2013 : $($objProc.ExitCode)"
    if ($objProc.ExitCode -notmatch "^(0|1638|3010)$") {
        Write-Host "[ERREUR] VC++ 2013 : L'installation a échoué (code $($objProc.ExitCode))." -ForegroundColor Red
        pause
    } else {
        if (Test-DllPresence @('msvcr120.dll','msvcp120.dll')) {
            Write-Host "[OK] VC++ 2013 : DLL présentes."
        } else {
            Write-Host "[WARNING] VC++ 2013 : DLL manquantes."
        }
        if (Test-ProgramInstalled 'Visual C++ 2013') {
            Write-Host "[OK] VC++ 2013 détecté dans les programmes installés."
        } else {
            Write-Host "[WARNING] VC++ 2013 NON détecté dans les programmes installés." -ForegroundColor Yellow
        }
    }
}


# --- UCMA 4.0 ---
# Recherche plus large pour UCMA dans le registre
function Test-UCMAInstalled {
    $paths = @(
        'HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall',
        'HKLM:\SOFTWARE\WOW6432Node\Microsoft\Windows\CurrentVersion\Uninstall'
    )
    foreach ($regPath in $paths) {
        $apps = Get-ChildItem $regPath | ForEach-Object {
            Get-ItemProperty $_.PSPath
        }
        foreach ($app in $apps) {
            if ($app.DisplayName -and ($app.DisplayName -like '*UCMA*' -or $app.DisplayName -like '*Unified Communications Managed API*')) {
                return $true
            }
        }
    }
    return $false
}

if (Test-UCMAInstalled) {
    Write-Host "[SKIP] UCMA déjà installé. Installation ignorée."
} else {
    Write-Host "[Core] Installation de UCMA 4.0 Runtime..." -ForegroundColor Cyan
    $objProc = Start-Process -FilePath $Ucma -ArgumentList "/passive /norestart" -NoNewWindow -PassThru -Wait
    Write-Host "Code retour UCMA : $($objProc.ExitCode)"
    if ($objProc.ExitCode -ne 0) {
        Write-Host "[ERREUR] UCMA : L'installation a échoué (code $($objProc.ExitCode)). Vérifiez le Setup.exe ou le journal d'installation." -ForegroundColor Red
        pause
    } else {
        Write-Host "[OK] UCMA installé."
        if (Test-UCMAInstalled) {
            Write-Host "[OK] UCMA détecté dans les programmes installés."
        } else {
            Write-Host "[WARNING] UCMA NON détecté dans les programmes installés." -ForegroundColor Yellow
        }
    }
}

# --- IIS URL Rewrite ---
if (Test-ProgramInstalled 'IIS URL Rewrite') {
    Write-Host "[SKIP] IIS URL Rewrite déjà installé. Installation ignorée."
} else {
    Write-Host "[Core] Installation de IIS URL Rewrite 2.1..." -ForegroundColor Cyan
    $objProc = Start-Process -FilePath $Rewrite -ArgumentList "/passive" -PassThru -wait
    Write-Host "Code retour IIS Rewrite : $($objProc.ExitCode)"
    if ($objProc.ExitCode -notmatch "^(0|1638|3010)$") {
        Write-Host "[ERREUR] IIS Rewrite : L'installation a échoué (code $($objProc.ExitCode))." -ForegroundColor Red
        pause
    } else {
        Write-Host "[OK] IIS Rewrite installé."
        if (Test-ProgramInstalled 'IIS URL Rewrite') {
            Write-Host "[OK] IIS URL Rewrite détecté dans les programmes installés."
        } else {
            Write-Host "[WARNING] IIS URL Rewrite NON détecté dans les programmes installés." -ForegroundColor Yellow
        }
    }
}


# Configuration TLS .NET recommandée avec vérification des clés
Write-Host "Configuration TLS .NET..."
$regKeys = @(
    @{ Path = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\.NETFramework\v2.0.50727"; Names = @("SchUseStrongCrypto", "SystemDefaultTlsVersions") },
    @{ Path = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\WOW6432Node\Microsoft\.NETFramework\v2.0.50727"; Names = @("SchUseStrongCrypto", "SystemDefaultTlsVersions") },
    @{ Path = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\.NETFramework\v4.0.30319"; Names = @("SchUseStrongCrypto", "SystemDefaultTlsVersions") },
    @{ Path = "Registry::HKEY_LOCAL_MACHINE\SOFTWARE\WOW6432Node\Microsoft\.NETFramework\v4.0.30319"; Names = @("SchUseStrongCrypto", "SystemDefaultTlsVersions") }
)
foreach ($reg in $regKeys) {
    foreach ($name in $reg.Names) {
        try {
            $current = Get-ItemProperty -Path $reg.Path -Name $name -ErrorAction SilentlyContinue
            if ($null -ne $current) {
                Set-ItemProperty -Path $reg.Path -Name $name -Value 1
                Write-Host "[UPDATE] $name mis à jour dans $($reg.Path)" -ForegroundColor Yellow
            } else {
                New-ItemProperty -Path $reg.Path -Name $name -PropertyType Dword -Value 1 | Out-Null
                Write-Host "[CREATE] $name créé dans $($reg.Path)" -ForegroundColor Green
            }
        } catch {
            Write-Host "[ERREUR] Impossible d'accéder à $($reg.Path)" -ForegroundColor Red
        }
    }
}

Write-Host "Installation des prérequis terminée." -ForegroundColor Cyan